<?php
header('Content-Type: application/json; charset=utf-8');
require_once 'config.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Função genérica para enviar mensagens JSON
function send_json_message($message, $class = 'text-red-400', $code = 400) {
    http_response_code($code);
    echo json_encode([
        'html' => "<div class='p-8 text-center $class'>$message</div>",
        'sessao_ativa' => false,
        'segundos_iniciais' => 0
    ]);
    exit;
}

if (!isset($_SESSION['loggedin']) || strtolower(trim($_SESSION['tipo'] ?? '')) !== 'aluno') {
    send_json_message("Acesso não autorizado.", 'text-red-400', 403);
}

if (!isset($_GET['data']) || !DateTime::createFromFormat('Y-m-d', $_GET['data'])) {
    send_json_message("Parâmetro de data inválido ou ausente.");
}

$id_aluno = $_SESSION['id'];
$data_selecionada = $_GET['data'];
$is_today = ($data_selecionada == date('Y-m-d'));

try {
    $treino_principal = null;
    $segundos_iniciais = 0;
    $mensagem_contexto = ''; // Para informar ao usuário por que ele está vendo um treino específico

    // --- NOVA LÓGICA DE PRIORIDADES ---

    // PRIORIDADE 1: Buscar sessão ATIVA para a data selecionada
    $stmt_ativo = $conexao->prepare("SELECT t.id, t.nome_treino, t.musculos_focados, st.hora_inicio FROM treinos t JOIN sessoes_treino st ON t.id = st.id_treino WHERE st.id_aluno = ? AND st.data_sessao = ? AND st.hora_fim IS NULL LIMIT 1");
    $stmt_ativo->bind_param("is", $id_aluno, $data_selecionada);
    $stmt_ativo->execute();
    $dados_treino_ativo = $stmt_ativo->get_result()->fetch_assoc();
    $stmt_ativo->close();

    if ($dados_treino_ativo) {
        $treino_principal = $dados_treino_ativo;
        if (!empty($dados_treino_ativo['hora_inicio'])) {
            $hora_inicio = new DateTime($dados_treino_ativo['hora_inicio']);
            $agora = new DateTime();
            $segundos_iniciais = max(0, $agora->getTimestamp() - $hora_inicio->getTimestamp());
        }
    }

    // Se não há treino ativo, continua a verificação
    if (!$treino_principal) {
        // PRIORIDADE 2: Buscar sessão CONCLUÍDA para a data selecionada
        $stmt_concluido = $conexao->prepare("SELECT t.nome_treino, t.musculos_focados FROM treinos t JOIN sessoes_treino st ON t.id = st.id_treino WHERE st.id_aluno = ? AND st.data_sessao = ? AND st.hora_fim IS NOT NULL LIMIT 1");
        $stmt_concluido->bind_param("is", $id_aluno, $data_selecionada);
        $stmt_concluido->execute();
        $treino_concluido = $stmt_concluido->get_result()->fetch_assoc();
        $stmt_concluido->close();

        if ($treino_concluido) {
            $nome_treino_concluido = htmlspecialchars($treino_concluido['musculos_focados'] ?: $treino_concluido['nome_treino']);
            $mensagem = "<i class='fas fa-check-circle fa-3x mb-4'></i><br>Você já concluiu o treino deste dia: <br><strong>$nome_treino_concluido</strong><br>";
            send_json_message($mensagem, 'text-green-400', 200);
        }

        // PRIORIDADE 3: Buscar TREINO PENDENTE de dias anteriores
        $data_inicio_busca = date('Y-m-d', strtotime('-7 days')); // Busca pendências na última semana
        $stmt_pendente = $conexao->prepare("
            SELECT DISTINCT t.id, t.nome_treino, t.musculos_focados
            FROM sessoes_treino st
            JOIN treinos t ON st.id_treino_substituido = t.id
            WHERE st.id_aluno = ? 
              AND st.id_treino_substituido IS NOT NULL
              AND st.data_sessao >= ?
              AND NOT EXISTS (
                  SELECT 1 FROM sessoes_treino st_done
                  WHERE st_done.id_aluno = st.id_aluno
                    AND st_done.id_treino = st.id_treino_substituido
              )
            LIMIT 1
        ");
        $stmt_pendente->bind_param("is", $id_aluno, $data_inicio_busca);
        $stmt_pendente->execute();
        $treino_pendente = $stmt_pendente->get_result()->fetch_assoc();
        $stmt_pendente->close();

        if ($treino_pendente) {
            $treino_principal = $treino_pendente;
            $mensagem_contexto = "<div class='bg-yellow-800 text-yellow-200 p-3 rounded-lg text-center mb-4 text-sm'>Este é um treino pendente que foi substituído em um dia anterior.</div>";
        }
    }

    // Se ainda não achou nenhum treino principal (nem ativo, nem pendente)
    if (!$treino_principal) {
        // PRIORIDADE 4: Buscar treino AGENDADO para o dia
        $dia_semana_selecionado = date('l', strtotime($data_selecionada));
        $dias_map_en_pt = ['Sunday' => 'Domingo', 'Monday' => 'Segunda-feira', 'Tuesday' => 'Terça-feira', 'Wednesday' => 'Quarta-feira', 'Thursday' => 'Quinta-feira', 'Friday' => 'Sexta-feira', 'Saturday' => 'Sábado'];
        $dia_semana_pt = $dias_map_en_pt[$dia_semana_selecionado];

        $stmt_agendado = $conexao->prepare("SELECT id, nome_treino, musculos_focados FROM treinos WHERE id_aluno = ? AND dia_semana = ? AND status = 'ativo'");
        $stmt_agendado->bind_param("is", $id_aluno, $dia_semana_pt);
        $stmt_agendado->execute();
        $treino_agendado = $stmt_agendado->get_result()->fetch_assoc();
        $stmt_agendado->close();

        if ($treino_agendado) {
            // Verificação extra: este treino agendado já foi feito como substituto em outro dia?
            $stmt_ja_feito = $conexao->prepare("SELECT 1 FROM sessoes_treino WHERE id_aluno = ? AND id_treino = ? AND data_sessao >= ? LIMIT 1");
            $data_inicio_busca = date('Y-m-d', strtotime('-7 days'));
            $stmt_ja_feito->bind_param("iis", $id_aluno, $treino_agendado['id'], $data_inicio_busca);
            $stmt_ja_feito->execute();
            $ja_foi_feito = $stmt_ja_feito->get_result()->num_rows > 0;
            $stmt_ja_feito->close();

            if (!$ja_foi_feito) {
                $treino_principal = $treino_agendado;
            }
        }
    }

    // Se, depois de toda a lógica, não houver treino, exibe a mensagem
    if (!$treino_principal) {
        send_json_message("Nenhum treino agendado ou pendente para este dia.", 'text-gray-400', 200);
    }
    
    // --- Continua com a renderização do treino encontrado ---
    ob_start();

    $treino_id_principal = $treino_principal['id'];
    $sessao_ativa = $is_today && ($dados_treino_ativo);
    
    // O resto do código para buscar exercícios, histórico, etc., continua igual
    $stmt_ex = $conexao->prepare("SELECT te.id as id_treino_exercicio, e.nome_exercicio, te.series, te.repeticoes, te.carga_inicial FROM treino_exercicios te JOIN exercicios e ON te.id_exercicio = e.id WHERE te.id_treino = ?");
    $stmt_ex->bind_param("i", $treino_id_principal);
    $stmt_ex->execute();
    $exercicios = $stmt_ex->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt_ex->close();

    $stmt_hist = $conexao->prepare("SELECT id_treino_exercicio, carga_utilizada FROM historico_treino WHERE id_aluno = ? AND data_checkin = ?");
    $stmt_hist->bind_param("is", $id_aluno, $data_selecionada);
    $stmt_hist->execute();
    $exercicios_finalizados_raw = $stmt_hist->get_result()->fetch_all(MYSQLI_ASSOC);
    $exercicios_finalizados = array_column($exercicios_finalizados_raw, 'carga_utilizada', 'id_treino_exercicio');
    $stmt_hist->close();

    $total_exercicios = count($exercicios);
    $exercicios_completos = count($exercicios_finalizados);
    $progresso_percentual = ($total_exercicios > 0) ? ($exercicios_completos / $total_exercicios) * 100 : 0;

    $stmt_outros = $conexao->prepare("SELECT id, nome_treino, musculos_focados FROM treinos WHERE id_aluno = ? AND id != ? AND status = 'ativo'");
    $stmt_outros->bind_param("ii", $id_aluno, $treino_id_principal);
    $stmt_outros->execute();
    $outros_treinos = $stmt_outros->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt_outros->close();

    // Inclui o template que vai renderizar o HTML
    // Note que o template agora pode usar a variável $mensagem_contexto
    include 'template_treino_do_dia.php';
    $html = ob_get_clean();

    echo json_encode([
        'html' => $html, 
        'sessao_ativa' => $sessao_ativa,
        'segundos_iniciais' => $segundos_iniciais 
    ]);

} catch (Exception $e) {
    send_json_message("Erro no servidor ao processar o treino: " . $e->getMessage(), 'text-red-500', 500);
}
?>