<?php
require_once 'config.php';

// É crucial iniciar a sessão para poder usar a variável $_SESSION
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

header('Content-Type: application/json');

// Verifica se está logado e é aluno
if (!isset($_SESSION['loggedin']) || $_SESSION['tipo'] !== 'aluno') {
    http_response_code(403); // Proibido
    echo json_encode(['success' => false, 'message' => 'Usuário não autenticado']);
    exit;
}

$id_aluno = $_SESSION['id'];
$acao = $_GET['acao'] ?? '';

if ($acao === 'iniciar') {
    // Recebe os dados JSON do POST
    $input = json_decode(file_get_contents('php://input'), true);
    $id_treino = isset($input['treino_id']) ? (int)$input['treino_id'] : 0;
    // Pega a data da sessão enviada pelo frontend, não usa mais a data fixa de hoje
    $data_sessao = $input['data_sessao'] ?? null;

    if ($id_treino <= 0 || !$data_sessao) {
        http_response_code(400); // Requisição inválida
        echo json_encode(['success' => false, 'message' => 'ID do treino ou data da sessão inválidos']);
        exit;
    }

    // --- LÓGICA MELHORADA E NOVA ---

    try {
        // Verifica se já existe QUALQUER sessão ativa hoje para este aluno
        $stmt = $conexao->prepare("SELECT id FROM sessoes_treino WHERE id_aluno = ? AND data_sessao = ? AND hora_fim IS NULL");
        $stmt->bind_param("is", $id_aluno, $data_sessao);
        $stmt->execute();
        $result = $stmt->get_result();
        $stmt->close();

        if ($result->num_rows > 0) {
            // Sessão já existe, não precisa fazer nada
            echo json_encode(['success' => true, 'message' => 'Uma sessão já está ativa para este dia.']);
            exit;
        }

        // --- INÍCIO DA LÓGICA DE SUBSTITUIÇÃO ---
        $id_treino_substituido = null;

        // 1. Descobrir qual era o treino originalmente agendado para a data da sessão
        $dia_semana_selecionado = date('l', strtotime($data_sessao));
        $dias_map_en_pt = ['Sunday' => 'Domingo', 'Monday' => 'Segunda-feira', 'Tuesday' => 'Terça-feira', 'Wednesday' => 'Quarta-feira', 'Thursday' => 'Quinta-feira', 'Friday' => 'Sexta-feira', 'Saturday' => 'Sábado'];
        $dia_semana_pt = $dias_map_en_pt[$dia_semana_selecionado];

        $stmt_agendado = $conexao->prepare("SELECT id FROM treinos WHERE id_aluno = ? AND dia_semana = ? AND status = 'ativo'");
        $stmt_agendado->bind_param("is", $id_aluno, $dia_semana_pt);
        $stmt_agendado->execute();
        $resultado_agendado = $stmt_agendado->get_result()->fetch_assoc();
        $id_treino_agendado_original = $resultado_agendado['id'] ?? null;
        $stmt_agendado->close();

        // 2. Se o treino a iniciar é diferente do agendado, guardamos o ID do agendado
        if ($id_treino_agendado_original && $id_treino != $id_treino_agendado_original) {
            $id_treino_substituido = $id_treino_agendado_original;
        }
        // --- FIM DA LÓGICA DE SUBSTITUIÇÃO ---

        // Insere nova sessão com a coluna 'id_treino_substituido' preenchida (se aplicável)
        $stmt = $conexao->prepare("INSERT INTO sessoes_treino (id_aluno, id_treino, id_treino_substituido, data_sessao, hora_inicio) VALUES (?, ?, ?, ?, NOW())");
        $stmt->bind_param("iiss", $id_aluno, $id_treino, $id_treino_substituido, $data_sessao);
        $executou = $stmt->execute();
        $stmt->close();

        if ($executou) {
            echo json_encode(['success' => true, 'message' => 'Sessão iniciada com sucesso']);
        } else {
            throw new Exception("Erro ao inserir a sessão no banco de dados.");
        }
    } catch (Exception $e) {
        http_response_code(500); // Erro interno do servidor
        echo json_encode(['success' => false, 'message' => 'Erro no servidor: ' . $e->getMessage()]);
    }
    exit;
}

// Ação não reconhecida
http_response_code(400);
echo json_encode(['success' => false, 'message' => 'Ação inválida']);
exit;