<?php
require_once '../config.php';
if (!isset($_SESSION['loggedin']) || $_SESSION['tipo'] !== 'aluno') {
    header('Location: ../index.html'); exit;
}
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cardio HIIT</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <script src="https://cdnjs.cloudflare.com/ajax/libs/tone/14.7.77/Tone.js"></script>
</head>
<body class="bg-gray-900 text-white pb-20">
    <header class="bg-white shadow-md p-4 flex justify-between items-center">
        <h1 class="text-2xl font-bold text-black">CARDIO HIIT</h1>
        <a href="javascript:history.back()" class="bg-blue-500 hover:bg-blue-600 text-white font-bold py-2 px-4 rounded-lg">Voltar</a>
    </header>
    <main class="container mx-auto p-4">
        <div id="selecao-treino" class="bg-gray-800 p-6 rounded-lg">
            <div class="mb-4">
                <label class="block mb-2 font-semibold">Duração:</label>
                <div class="flex gap-2">
                    <button class="btn-duracao flex-1 bg-gray-700 p-3 rounded-lg" data-duracao="10">10 min</button>
                    <button class="btn-duracao flex-1 bg-gray-700 p-3 rounded-lg" data-duracao="15">15 min</button>
                    <button class="btn-duracao flex-1 bg-gray-700 p-3 rounded-lg" data-duracao="20">20 min</button>
                </div>
            </div>
            <div>
                <label class="block mb-2 font-semibold">Intensidade:</label>
                <div class="flex gap-2">
                    <button class="btn-intensidade flex-1 bg-gray-700 p-3 rounded-lg" data-intensidade="Fácil">Iniciante</button>
                    <button class="btn-intensidade flex-1 bg-gray-700 p-3 rounded-lg" data-intensidade="Médio">Intermediario</button>
                    <button class="btn-intensidade flex-1 bg-gray-700 p-3 rounded-lg" data-intensidade="Rápido">Avançado</button>
                </div>
            </div>
            <div class="mt-6">
                <canvas id="protocoloChart"></canvas>
            </div>
            <button id="iniciar-cardio-btn" class="w-full bg-green-500 hover:bg-green-600 text-white font-bold py-3 rounded-lg mt-6 hidden">Iniciar Treino</button>
        </div>

        <div id="treino-ativo" class="hidden bg-gray-800 p-6 rounded-lg text-center">
            <p class="text-gray-400 text-lg">Tempo Restante do Segmento:</p>
            <p id="timer-segmento" class="text-7xl font-bold my-4">00:00</p>
            <p id="instrucao-velocidade" class="text-3xl font-semibold"></p>
            <div class="mt-6">
                <div class="relative pt-1">
                    <div class="flex mb-2 items-center justify-between">
                        <div><span class="text-xs font-semibold inline-block text-indigo-300">Progresso Total</span></div>
                        <div class="text-right"><span id="cronometro-total" class="text-xs font-semibold inline-block text-indigo-300">00:00</span></div>
                    </div>
                    <div class="overflow-hidden h-4 mb-4 text-xs flex rounded bg-gray-600">
                        <div id="progresso-total-barra" style="width: 0%" class="shadow-none flex flex-col text-center whitespace-nowrap text-white justify-center bg-indigo-500"></div>
                    </div>
                </div>
            </div>
            <div class="mt-6">
                <h3 class="text-lg font-semibold text-gray-400 mb-2">Próximos Segmentos</h3>
                <canvas id="proximoSegmentoChart"></canvas>
            </div>
            <div class="flex gap-4 mt-6">
                <button id="pause-resume-btn" class="flex-1 bg-yellow-500 hover:bg-yellow-600 text-white font-bold py-3 rounded-lg"><i class="fas fa-pause"></i> Pausar</button>
                <button id="end-workout-btn" class="flex-1 bg-red-500 hover:bg-red-600 text-white font-bold py-3 rounded-lg"><i class="fas fa-stop"></i> Encerrar</button>
            </div>
        </div>
    </main>
    <div id="confirm-modal" class="fixed inset-0 bg-black bg-opacity-75 flex items-center justify-center hidden z-50 p-4">
        <div class="bg-gray-700 p-8 rounded-lg text-center">
            <h3 class="text-xl font-bold mb-4">Atenção!</h3>
            <p class="mb-6">Tem a certeza que deseja encerrar o treino antes de o terminar?</p>
            <div class="flex justify-center gap-4">
                <button id="confirm-yes" class="bg-red-500 hover:bg-red-600 text-white font-bold py-2 px-6 rounded-lg">Sim, encerrar</button>
                <button id="confirm-no" class="bg-gray-500 hover:bg-gray-400 text-white font-bold py-2 px-6 rounded-lg">Cancelar</button>
            </div>
        </div>
    </div>
    <nav class="fixed bottom-0 left-0 right-0 bg-white border-t border-gray-200 flex justify-around p-4">
        <a href="dashboard_aluno.php" class="flex flex-col items-center text-gray-400 hover:text-indigo-400 w-1/3"><i class="fas fa-home fa-lg"></i><span class="text-xs mt-3">Início</span></a>
        <a href="treino_do_dia.php" class="flex flex-col items-center text-gray-400 hover:text-indigo-400 w-1/3"><i class="fas fa-dumbbell fa-lg"></i><span class="text-xs mt-3">Treino</span></a>
        <a href="perfil_aluno.php" class="flex flex-col items-center text-indigo-400 w-1/3"><i class="fas fa-user fa-lg"></i><span class="text-xs mt-3">Perfil</span></a>
    </nav>
    
<script>
document.addEventListener('DOMContentLoaded', function() {
    // VARIÁVEIS GLOBAIS
    let selectedDuracao = null;
    let selectedIntensidade = null;
    let protocoloAtual = null;
    let protocoloChart, proximoSegmentoChart;
    let intervalo;
    let isPaused = false;
    let distanciaTotal = 0;
    let tempoTotalCorrido = 0;

    // CONSTANTES E ELEMENTOS DO DOM
    const btnsDuracao = document.querySelectorAll('.btn-duracao');
    const btnsIntensidade = document.querySelectorAll('.btn-intensidade');
    const iniciarBtn = document.getElementById('iniciar-cardio-btn');
    const ctx = document.getElementById('protocoloChart').getContext('2d');
    const ctxProximo = document.getElementById('proximoSegmentoChart').getContext('2d');
    const pauseResumeBtn = document.getElementById('pause-resume-btn');
    const endWorkoutBtn = document.getElementById('end-workout-btn');
    const modal = document.getElementById('confirm-modal');
    const confirmYes = document.getElementById('confirm-yes');
    const confirmNo = document.getElementById('confirm-no');
    const WORKOUT_STATE_KEY = 'hiitWorkoutState';

    // SETUP DE ÁUDIO
    const synth = new Tone.Synth().toDestination();
    let isAudioReady = false;

    async function startAudioContext() {
        if (isAudioReady) return;
        try {
            await Tone.start();
            console.log("Contexto de áudio iniciado com sucesso!");
            isAudioReady = true;
        } catch (e) {
            console.error("Não foi possível iniciar o contexto de áudio", e);
        }
    }

    // FUNÇÕES DE SELEÇÃO E GRÁFICO INICIAL
    function updateSelection() {
        btnsDuracao.forEach(btn => btn.classList.toggle('bg-indigo-500', btn.dataset.duracao == selectedDuracao));
        btnsIntensidade.forEach(btn => btn.classList.toggle('bg-indigo-500', btn.dataset.intensidade == selectedIntensidade));
        if (selectedDuracao && selectedIntensidade) {
            fetchAndDrawChart();
        }
    }

    async function fetchAndDrawChart() {
        const response = await fetch(`api_get_hiit_protocolo.php?duracao=${selectedDuracao}&intensidade=${selectedIntensidade}`);
        protocoloAtual = await response.json();
        if (protocoloChart) protocoloChart.destroy();
        const labels = [],
            data = [];
        let tempoAcumulado = 0;
        protocoloAtual.forEach(seg => {
            const minutos = tempoAcumulado / 60;
            labels.push(`${minutos}'`);
            data.push(seg.velocidade);
            tempoAcumulado += seg.tempo;
        });
        protocoloChart = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Velocidade (km/h)',
                    data: data,
                    backgroundColor: 'rgba(99, 102, 241, 0.7)',
                    borderColor: 'rgba(99, 102, 241, 1)',
                    borderWidth: 1
                }]
            },
            options: {
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });
        // A LINHA QUE EU TINHA APAGADO ESTÁ DE VOLTA AQUI:
        iniciarBtn.classList.remove('hidden');
    }

    // FUNÇÃO GLOBAL DE FORMATAÇÃO
    function formatarTempo(s) {
        const minutos = Math.floor(s / 60).toString().padStart(2, '0');
        const segundos = (s % 60).toString().padStart(2, '0');
        return `${minutos}:${segundos}`;
    }

    // LÓGICA PRINCIPAL DO TREINO
    function iniciarTreinoHIIT(initialState = null) {
        let segmentoAtual, tempoRestanteSegmento;

        if (initialState) {
            protocoloAtual = initialState.protocolo;
            segmentoAtual = initialState.segmentoAtual;
            tempoRestanteSegmento = initialState.tempoRestanteSegmento;
            tempoTotalCorrido = initialState.tempoTotalCorrido;
            distanciaTotal = initialState.distanciaTotal;
            isPaused = initialState.isPaused;
            selectedDuracao = initialState.selectedDuracao;
            selectedIntensidade = initialState.selectedIntensidade;
            updateUI(tempoTotalCorrido, tempoRestanteSegmento, segmentoAtual, distanciaTotal, isPaused);
        } else {
            segmentoAtual = 0;
            tempoRestanteSegmento = protocoloAtual[0].tempo;
            tempoTotalCorrido = 0;
            distanciaTotal = 0;
            isPaused = false;
        }

        drawProximoSegmentoChart();
        updateChartHighlight(segmentoAtual);

        function saveState() {
            const state = {
                protocolo: protocoloAtual,
                segmentoAtual,
                tempoRestanteSegmento,
                tempoTotalCorrido,
                distanciaTotal,
                isPaused,
                selectedDuracao,
                selectedIntensidade
            };
            localStorage.setItem(WORKOUT_STATE_KEY, JSON.stringify(state));
        }

        function updateUI(total, segmento, index, dist, paused) {
            const duracaoTotal = protocoloAtual.reduce((acc, seg) => acc + seg.tempo, 0);
            document.getElementById('timer-segmento').textContent = formatarTempo(segmento);
            document.getElementById('instrucao-velocidade').innerHTML = `Corra a <span class="text-indigo-400">${protocoloAtual[index].velocidade} km/h</span> - Distância: ${dist.toFixed(2)} km`;
            document.getElementById('progresso-total-barra').style.width = `${(total / duracaoTotal) * 100}%`;
            document.getElementById('cronometro-total').textContent = formatarTempo(total);
            pauseResumeBtn.innerHTML = paused ? '<i class="fas fa-play"></i> Retomar' : '<i class="fas fa-pause"></i> Pausar';
            pauseResumeBtn.classList.toggle('bg-yellow-500', !paused);
            pauseResumeBtn.classList.toggle('bg-green-500', paused);
        }

        function drawProximoSegmentoChart() {
            if (proximoSegmentoChart) proximoSegmentoChart.destroy();
            const labels = [], data = [];
            let tempoAcumulado = 0;
            protocoloAtual.forEach(seg => {
                const minutos = tempoAcumulado / 60;
                labels.push(`${minutos}'`);
                data.push(seg.velocidade);
                tempoAcumulado += seg.tempo;
            });
            proximoSegmentoChart = new Chart(ctxProximo, {
                type: 'bar',
                data: {
                    labels: labels,
                    datasets: [{
                        label: 'Velocidade (km/h)',
                        data: data
                    }]
                },
                options: {
                    scales: {
                        y: {
                            beginAtZero: true
                        }
                    }
                }
            });
        }

        function updateChartHighlight(segmentoAtivo) {
            if (!proximoSegmentoChart) return;
            const backgroundColors = protocoloAtual.map((seg, index) => {
                return index === segmentoAtivo ? 'rgba(239, 68, 68, 0.7)' : 'rgba(99, 102, 241, 0.7)';
            });
            proximoSegmentoChart.data.datasets[0].backgroundColor = backgroundColors;
            proximoSegmentoChart.update();
        }

        function proximoSegmento() {
            segmentoAtual++;
            if (segmentoAtual >= protocoloAtual.length) {
                clearInterval(intervalo);
                document.getElementById('instrucao-velocidade').textContent = "Treino Concluído!";
                document.getElementById('timer-segmento').textContent = "Parabéns!";
                synth.triggerAttackRelease("C6", "4n");
                salvarSessao(formatarTempo(tempoTotalCorrido), distanciaTotal);
                localStorage.removeItem(WORKOUT_STATE_KEY);
                return;
            }
            tempoRestanteSegmento = protocoloAtual[segmentoAtual].tempo;
            synth.triggerAttackRelease("C5", "8n");
            updateChartHighlight(segmentoAtual);
        }

        intervalo = setInterval(() => {
            if (isPaused) {
                saveState();
                return;
            }
            tempoRestanteSegmento--;
            tempoTotalCorrido++;
            distanciaTotal += (protocoloAtual[segmentoAtual].velocidade / 3600);
            updateUI(tempoTotalCorrido, tempoRestanteSegmento, segmentoAtual, distanciaTotal, isPaused);
            if (tempoRestanteSegmento <= 5 && tempoRestanteSegmento > 0) {
                synth.triggerAttackRelease("A4", "8n");
            }
            if (tempoRestanteSegmento < 0) {
                proximoSegmento();
            }
            saveState();
        }, 1000);
    }

    function resumeWorkout(state) {
        document.getElementById('selecao-treino').classList.add('hidden');
        document.getElementById('treino-ativo').classList.remove('hidden');
        iniciarTreinoHIIT(state);
    }

    async function salvarSessao(duracao, distancia) {
        if (distancia > 0) {
            await fetch('salvar_cardio_hiit.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    duracao: duracao,
                    distancia: distancia.toFixed(2),
                    protocolo: `${selectedDuracao} min - ${selectedIntensidade}`
                })
            });
        }
    }

    // EVENT LISTENERS
    btnsDuracao.forEach(btn => btn.addEventListener('click', () => {
        selectedDuracao = btn.dataset.duracao;
        updateSelection();
    }));
    btnsIntensidade.forEach(btn => btn.addEventListener('click', () => {
        selectedIntensidade = btn.dataset.intensidade;
        updateSelection();
    }));
    iniciarBtn.addEventListener('click', function() {
        startAudioContext();
        document.getElementById('selecao-treino').classList.add('hidden');
        document.getElementById('treino-ativo').classList.remove('hidden');
        iniciarTreinoHIIT();
    });
    pauseResumeBtn.addEventListener('click', () => {
        startAudioContext();
        isPaused = !isPaused;
    });
    endWorkoutBtn.addEventListener('click', () => {
        isPaused = true;
        modal.classList.remove('hidden');
    });
    confirmYes.addEventListener('click', async () => {
        clearInterval(intervalo);
        await salvarSessao(formatarTempo(tempoTotalCorrido), distanciaTotal);
        localStorage.removeItem(WORKOUT_STATE_KEY);
        window.location.href = '../perfil_aluno.php';
    });
    confirmNo.addEventListener('click', () => {
        isPaused = false;
        modal.classList.add('hidden');
    });

    // INICIALIZAÇÃO DA PÁGINA
    const savedStateJSON = localStorage.getItem(WORKOUT_STATE_KEY);
    if (savedStateJSON) {
        const savedState = JSON.parse(savedStateJSON);
        resumeWorkout(savedState);
    }
});
</script>


</body>
</html>