<?php
require_once '../config.php';
if (!isset($_SESSION['loggedin']) || $_SESSION['tipo'] !== 'personal') {
    header('Location: ../index.html'); exit;
}
$id_personal = $_SESSION['id'];
$alunos_stmt = $conexao->prepare("
    SELECT u.id, u.nome, (SELECT COUNT(*) FROM mensagens m WHERE m.id_remetente = u.id AND m.id_destinatario = ? AND m.lida = FALSE) as nao_lidas
    FROM usuarios u 
    WHERE u.id_personal = ? 
    ORDER BY u.nome
");
$alunos_stmt->bind_param("ii", $id_personal, $id_personal);
$alunos_stmt->execute();
$alunos = $alunos_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Chat com Alunos</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <script src="https://cdnjs.cloudflare.com/ajax/libs/tone/14.7.77/Tone.js"></script>
</head>
<body class="bg-gray-900 text-white flex flex-col h-screen">
    <header class="bg-gray-800 p-4 flex justify-between items-center">
        <div class="flex items-center gap-4">
            <button id="toggle-alunos-btn" class="md:hidden bg-gray-700 p-2 rounded-lg"><i class="fas fa-users"></i></button>
            <h1 class="text-xl font-bold">Chat com Alunos</h1>
        </div>
        <div>
            <button id="mute-btn" class="text-white mr-4"><i class="fas fa-volume-up"></i></button>
            <a href="../dashboard_personal.php" class="bg-cyan-500 hover:bg-cyan-600 text-gray-900 font-bold py-2 px-4 rounded-lg">Voltar</a>
        </div>
    </header>
    <div class="flex flex-1 overflow-hidden relative">
        <aside id="lista-alunos-sidebar" class="absolute md:relative h-full w-2/3 md:w-1/3 bg-gray-800 p-4 overflow-y-auto transition-transform transform -translate-x-full md:translate-x-0 z-20">
            <h2 class="text-lg font-semibold mb-4">Meus Alunos</h2>
            <ul id="lista-alunos-chat">
                <?php foreach ($alunos as $aluno): ?>
                    <li class="p-2 rounded hover:bg-gray-700 cursor-pointer chat-aluno-item flex justify-between items-center" data-aluno-id="<?php echo $aluno['id']; ?>" data-aluno-nome="<?php echo htmlspecialchars($aluno['nome']); ?>">
                        <span><?php echo htmlspecialchars($aluno['nome']); ?></span>
                        <?php if ($aluno['nao_lidas'] > 0): ?>
                            <span class="bg-blue-500 text-white text-xs font-bold rounded-full h-5 w-5 flex items-center justify-center"><?php echo $aluno['nao_lidas']; ?></span>
                        <?php endif; ?>
                    </li>
                <?php endforeach; ?>
            </ul>
        </aside>
        <main class="w-full md:flex-1 flex flex-col p-4">
            <div id="chat-window" class="flex-1 bg-gray-700 rounded-lg p-4 flex flex-col gap-4 overflow-y-auto mb-4">
                <p class="text-center text-gray-400">Selecione um aluno para iniciar a conversa.</p>
            </div>
            <form id="chat-form" class="flex gap-4">
                <input type="hidden" id="destinatario-id" value="">
                <input type="text" id="mensagem-input" class="flex-1 bg-gray-600 rounded-lg p-3" placeholder="Digite sua mensagem..." disabled>
                <button type="submit" id="enviar-btn" class="bg-green-500 hover:bg-green-600 font-bold py-3 px-5 rounded-lg" disabled><i class="fas fa-paper-plane"></i></button>
            </form>
        </main>
    </div>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const listaAlunos = document.querySelectorAll('.chat-aluno-item');
    const chatWindow = document.getElementById('chat-window');
    const chatForm = document.getElementById('chat-form');
    const mensagemInput = document.getElementById('mensagem-input');
    const enviarBtn = document.getElementById('enviar-btn');
    const destinatarioIdInput = document.getElementById('destinatario-id');
    const muteBtn = document.getElementById('mute-btn');
    const toggleAlunosBtn = document.getElementById('toggle-alunos-btn');
    const sidebar = document.getElementById('lista-alunos-sidebar');
    let intervaloRefresh;
    let totalMensagensAtual = 0;
    const synth = new Tone.Synth().toDestination();
    let isMuted = false;

    async function carregarMensagens(alunoId) {
        const response = await fetch(`api_get_conversa.php?outro_usuario_id=${alunoId}`);
        const data = await response.json();
        
        if (data.mensagens.length > totalMensagensAtual && totalMensagensAtual > 0) {
            const ultimaMsg = data.mensagens[data.mensagens.length - 1];
            if (ultimaMsg.id_remetente != data.id_logado && !isMuted) {
                synth.triggerAttackRelease("C5", "8n");
            }
        }
        totalMensagensAtual = data.mensagens.length;

        chatWindow.innerHTML = '';
        data.mensagens.forEach(msg => {
            const alinhamento = msg.id_remetente == data.id_logado ? 'items-end' : 'items-start';
            const corBalao = msg.id_remetente == data.id_logado ? 'bg-cyan-800' : 'bg-gray-600';
            chatWindow.innerHTML += `
                <div class="w-full flex flex-col ${alinhamento}">
                    <div class="p-3 rounded-lg max-w-xs ${corBalao}">
                        <p class="font-bold text-sm">${msg.nome_remetente}</p>
                        <p>${msg.mensagem}</p>
                        <p class="text-xs text-gray-400 text-right mt-1">${msg.data_formatada}</p>
                    </div>
                </div>
            `;
        });
        chatWindow.scrollTop = chatWindow.scrollHeight;
    }

    listaAlunos.forEach(item => {
        item.addEventListener('click', function() {
            const alunoId = this.dataset.alunoId;
            const alunoNome = this.dataset.alunoNome;
            
            destinatarioIdInput.value = alunoId;
            mensagemInput.disabled = false;
            enviarBtn.disabled = false;
            mensagemInput.placeholder = `Mensagem para ${alunoNome}...`;
            totalMensagensAtual = 0;

            const notificationBadge = this.querySelector('span:last-child');
            if (notificationBadge) {
                notificationBadge.remove();
            }

            if (window.innerWidth < 768) { // Se for tela pequena, esconde a lista
                sidebar.classList.add('-translate-x-full');
            }

            clearInterval(intervaloRefresh);
            carregarMensagens(alunoId);
            intervaloRefresh = setInterval(() => carregarMensagens(alunoId), 3000);
        });
    });

    chatForm.addEventListener('submit', async function(e) {
        e.preventDefault();
        const destinatarioId = destinatarioIdInput.value;
        const mensagem = mensagemInput.value.trim();
        if (!destinatarioId || !mensagem) return;

        await fetch('api_enviar_mensagem.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ destinatario_id: destinatarioId, mensagem: mensagem })
        });
        mensagemInput.value = '';
        carregarMensagens(destinatarioId);
    });

    muteBtn.addEventListener('click', function() {
        isMuted = !isMuted;
        this.innerHTML = isMuted ? '<i class="fas fa-volume-mute"></i>' : '<i class="fas fa-volume-up"></i>';
    });
    
    toggleAlunosBtn.addEventListener('click', () => {
        sidebar.classList.toggle('-translate-x-full');
    });
});
</script>
</body>
</html>

