<?php
require_once 'config.php';
if (!isset($_SESSION['loggedin']) || !in_array($_SESSION['tipo'], ['adm', 'personal'])) {
    header('Location: index.html'); exit;
}

$aluno_selecionado_id = isset($_GET['aluno_id']) ? (int)$_GET['aluno_id'] : 0;
if ($aluno_selecionado_id === 0) {
    die("ID do aluno não fornecido.");
}

// Busca o nome do aluno
$aluno_stmt = $conexao->prepare("SELECT nome FROM usuarios WHERE id = ?");
$aluno_stmt->bind_param("i", $aluno_selecionado_id);
$aluno_stmt->execute();
$aluno_nome = $aluno_stmt->get_result()->fetch_assoc()['nome'];
$aluno_stmt->close();

// Busca todos os treinos e exercícios do aluno de uma vez
$treinos_query = $conexao->prepare("
    SELECT 
        t.id as treino_id, t.nome_treino, t.dia_semana, t.musculos_focados, t.status,
        te.id as treino_exercicio_id, e.nome_exercicio, te.carga_inicial,
        (SELECT ht.carga_utilizada 
         FROM historico_treino ht 
         WHERE ht.id_treino_exercicio = te.id 
         ORDER BY ht.data_checkin DESC, ht.id DESC 
         LIMIT 1) as ultima_carga
    FROM treinos t
    LEFT JOIN treino_exercicios te ON t.id = te.id_treino
    LEFT JOIN exercicios e ON te.id_exercicio = e.id
    WHERE t.id_aluno = ?
    ORDER BY FIELD(t.dia_semana, 'Segunda-feira', 'Terça-feira', 'Quarta-feira', 'Quinta-feira', 'Sexta-feira', 'Sábado', 'Domingo'), t.data_criacao DESC
");
$treinos_query->bind_param("i", $aluno_selecionado_id);
$treinos_query->execute();
$result = $treinos_query->get_result();
$treinos_formatados = [];
while ($row = $result->fetch_assoc()) {
    $treinos_formatados[$row['treino_id']]['info'] = [
        'nome_treino' => $row['nome_treino'],
        'dia_semana' => $row['dia_semana'],
        'musculos_focados' => $row['musculos_focados'],
        'status' => $row['status']
    ];
    if ($row['treino_exercicio_id']) {
        $treinos_formatados[$row['treino_id']]['exercicios'][] = [
            'nome_exercicio' => $row['nome_exercicio'],
            'carga_inicial' => $row['carga_inicial'],
            'ultima_carga' => $row['ultima_carga']
        ];
    }
}
$treinos_query->close();
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <title>Histórico de Treinos</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
</head>
<body class="bg-gray-900 text-white">
    <header class="bg-gray-800 p-4 flex justify-between items-center">
        <h1 class="text-xl font-bold">Histórico de Treinos de <?php echo htmlspecialchars($aluno_nome); ?></h1>
        <a href="gerenciar_meus_alunos.php" class="bg-cyan-500 hover:bg-cyan-600 text-gray-900 font-bold py-2 px-4 rounded-lg">Voltar</a>
    </header>
    <main class="p-4">
        <div class="bg-gray-800 p-6 rounded-lg">
            <div id="lista-treinos-atuais" class="space-y-4 max-h-[80vh] overflow-y-auto pr-2">
                <?php if (empty($treinos_formatados)): ?>
                    <p class="text-center text-gray-400">Nenhum plano de treino encontrado para este aluno.</p>
                <?php else: ?>
                    <?php foreach ($treinos_formatados as $treino_id => $treino_data): ?>
                        <?php $treino = $treino_data['info']; $isAtivo = $treino['status'] === 'ativo'; ?>
                        <div class="bg-gray-700 p-3 rounded">
                            <div class="flex justify-between items-center">
                                <span class="font-bold"><?php echo htmlspecialchars($treino['musculos_focados'] ?: $treino['nome_treino']); ?> (<?php echo $treino['dia_semana']; ?>)</span>
                                <div class="flex items-center gap-4">
                                    <label class="relative inline-flex items-center cursor-pointer">
                                        <input type="checkbox" class="sr-only peer status-toggle" data-treino-id="<?php echo $treino_id; ?>" <?php if ($isAtivo) echo 'checked'; ?>>
                                        <div class="w-11 h-6 bg-gray-600 rounded-full peer peer-checked:after:translate-x-full after:content-[''] after:absolute after:top-0.5 after:left-[2px] after:bg-white after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-green-500"></div>
                                    </label>
                                    <button class="remover-treino text-red-500 hover:text-red-400" data-treino-id="<?php echo $treino_id; ?>"><i class="fas fa-trash"></i></button>
                                </div>
                            </div>
                            <table class="w-full text-left text-sm mt-2">
                                <thead class="bg-gray-600"><tr><th class="p-2 font-semibold">Exercício</th><th class="p-2 font-semibold">Carga Inicial</th><th class="p-2 font-semibold">Última Carga</th></tr></thead>
                                <tbody>
                                    <?php if (!empty($treino_data['exercicios'])): ?>
                                        <?php foreach ($treino_data['exercicios'] as $ex): ?>
                                            <tr class="border-b border-gray-600">
                                                <td class="p-2"><?php echo htmlspecialchars($ex['nome_exercicio']); ?></td>
                                                <td class="p-2"><?php echo htmlspecialchars($ex['carga_inicial'] ?? 'N/A'); ?> kg</td>
                                                <td class="p-2"><?php echo $ex['ultima_carga'] ? htmlspecialchars($ex['ultima_carga']) . ' kg' : 'Nenhum check-in'; ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php else: ?>
                                        <tr><td colspan="3" class="p-2 text-center text-gray-400">Nenhum exercício cadastrado.</td></tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </main>
<script>
document.addEventListener('DOMContentLoaded', function () {
    const listaTreinosAtuaisEl = document.getElementById('lista-treinos-atuais');

    listaTreinosAtuaisEl.addEventListener('change', async function(e) {
        if (e.target.classList.contains('status-toggle')) {
            const treinoId = e.target.dataset.treinoId;
            const novoStatus = e.target.checked ? 'ativo' : 'inativo';
            const response = await fetch('api_update_treino_status.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ treino_id: treinoId, status: novoStatus })
            });
            const result = await response.json();
            if (!result.success) {
                alert('Erro ao atualizar status: ' + result.message);
                e.target.checked = !e.target.checked;
            }
        }
    });

    listaTreinosAtuaisEl.addEventListener('click', async function(e) {
        const deleteButton = e.target.closest('.remover-treino');
        if (deleteButton) {
            const treinoId = deleteButton.dataset.treinoId;
            if (confirm('Tem certeza que deseja excluir este plano de treino?')) {
                const response = await fetch('api_delete_treino.php', { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ treino_id: treinoId }) });
                const result = await response.json();
                if (result.success) {
                    window.location.reload(); // Recarrega a página para ver a mudança
                } else {
                    alert('Erro ao excluir treino: ' + result.message);
                }
            }
        }
    });
});
</script>
</body>
</html>